import {
    includes
} from '#core/types/string';

import {
    dev
} from '#utils/log';

/**
 * This method wraps around window's open method. It first tries to execute
 * `open` call with the provided target and if it fails, it retries the call
 * with the `_top` target. This is necessary given that in some embedding
 * scenarios, such as iOS' WKWebView, navigation to `_blank` and other targets
 * is blocked by default.
 *
 * @param {!Window} win
 * @param {string} url
 * @param {string} target
 * @param {string=} opt_features
 * @return {?Window}
 */
export function openWindowDialog(win, url, target, opt_features) {
    // Try first with the specified target. If we're inside the WKWebView or
    // a similar environments, this method is expected to fail by default for
    // all targets except `_top`.
    let res;
    try {
        res = win.open(url, target, opt_features);
    } catch (e) {
        dev().error('DOM', 'Failed to open url on target: ', target, e);
    }

    // Then try with `_top` target.
    if (!res && target != '_top' && !includes(opt_features || '', 'noopener')) {
        res = win.open(url, '_top');
    }
    return res;
}